/********************************************************************
 FileName       : main.c
 Dependencies   : See INCLUDES section
 Processor      : PIC18, PIC24, and PIC32 USB Microcontrollers
 Hardware       : This demo is natively intended to be used on Microchip USB demo
 		  boards supported by the MCHPFSUSB stack.  See release notes for
 		  support matrix.  This demo can be modified for use on other hardware
 		  platforms.
 Complier       : Microchip C18 (for PIC18), C30 (for PIC24), C32 (for PIC32)
 Company        : Microchip Technology, Inc.

 Software License Agreement:

 The software supplied herewith by Microchip Technology Incorporated
 (the “Company”) for its PIC® Microcontroller is intended and
 supplied to you, the Company’s customer, for use solely and
 exclusively on Microchip PIC Microcontroller products. The
 software is owned by the Company and/or its supplier, and is
 protected under applicable copyright laws. All rights are reserved.
 Any use in violation of the foregoing restrictions may subject the
 user to criminal sanctions under applicable laws, as well as to
 civil liability for the breach of the terms and conditions of this
 license.

 THIS SOFTWARE IS PROVIDED IN AN “AS IS” CONDITION. NO WARRANTIES,
 WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.

********************************************************************
 File Description:

 Change History:
  Rev   Description
  1.0   Initial release
  2.1   Updated for simplicity and to use common
                     coding style
********************************************************************/
#include "USB/usb.h"
#include "USB/usb_host_msd.h"
#include "USB/usb_host_msd_scsi.h"
#include "MDD File System/FSIO.h"

// *****************************************************************************
// *****************************************************************************
// Configuration Bits
// *****************************************************************************
// *****************************************************************************

#if defined(ET_BASE_PIC32MX)
    #include "Configuration base-pic32mx.h"
#elif defined __C30__ || defined __XC16__
    #if defined(__PIC24FJ256GB110__)
        _CONFIG2(FNOSC_PRIPLL & POSCMOD_HS & PLL_96MHZ_ON & PLLDIV_DIV2 & IESO_OFF) // Primary HS OSC with PLL, USBPLL /2
        _CONFIG1(JTAGEN_OFF & FWDTEN_OFF & ICS_PGx2)   // JTAG off, watchdog timer off
    #elif defined(__PIC24FJ64GB004__)
        _CONFIG1(WDTPS_PS1 & FWPSA_PR32 & WINDIS_OFF & FWDTEN_OFF & ICS_PGx1 & GWRP_OFF & GCP_OFF & JTAGEN_OFF)
        _CONFIG2(POSCMOD_HS & I2C1SEL_PRI & IOL1WAY_OFF & OSCIOFNC_ON & FCKSM_CSDCMD & FNOSC_PRIPLL & PLL96MHZ_ON & PLLDIV_DIV2 & IESO_OFF)
        _CONFIG3(WPFP_WPFP0 & SOSCSEL_SOSC & WUTSEL_LEG & WPDIS_WPDIS & WPCFG_WPCFGDIS & WPEND_WPENDMEM)
        _CONFIG4(DSWDTPS_DSWDTPS3 & DSWDTOSC_LPRC & RTCOSC_SOSC & DSBOREN_OFF & DSWDTEN_OFF)
    #elif defined(__PIC24FJ256GB106__)
        _CONFIG1( JTAGEN_OFF & GCP_OFF & GWRP_OFF & FWDTEN_OFF & ICS_PGx2) 
        _CONFIG2( PLL_96MHZ_ON & IESO_OFF & FCKSM_CSDCMD & OSCIOFNC_OFF & POSCMOD_HS & FNOSC_PRIPLL & PLLDIV_DIV3 & IOL1WAY_ON)
    #elif defined(__PIC24FJ256DA210__) || defined(__PIC24FJ256GB210__)
        _CONFIG1(FWDTEN_OFF & ICS_PGx2 & GWRP_OFF & GCP_OFF & JTAGEN_OFF)
        _CONFIG2(POSCMOD_HS & IOL1WAY_ON & OSCIOFNC_ON & FCKSM_CSDCMD & FNOSC_PRIPLL & PLL96MHZ_ON & PLLDIV_DIV2 & IESO_OFF)
    #elif defined(__dsPIC33EP512MU810__)||defined(__PIC24EP512GU810__)
        _FOSCSEL(FNOSC_FRC);
        _FOSC(FCKSM_CSECMD & OSCIOFNC_OFF & POSCMD_XT);
        _FWDT(FWDTEN_OFF);
    #elif defined(__PIC24FJ64GB502__)
        _CONFIG1(WDTPS_PS1 & FWPSA_PR32 & WINDIS_OFF & FWDTEN_OFF & ICS_PGx1 & GWRP_OFF & GCP_OFF & JTAGEN_OFF)
	_CONFIG2(I2C1SEL_PRI & IOL1WAY_OFF & FCKSM_CSDCMD & FNOSC_PRIPLL & PLL96MHZ_ON & PLLDIV_DIV2 & IESO_OFF)
	_CONFIG3(WPFP_WPFP0 & SOSCSEL_SOSC & WUTSEL_LEG & WPDIS_WPDIS & WPCFG_WPCFGDIS & WPEND_WPENDMEM)
	_CONFIG4(DSWDTPS_DSWDTPS3 & DSWDTOSC_LPRC & RTCOSC_SOSC & DSBOREN_OFF & DSWDTEN_OFF)   
    #endif

//ET-BASE PIC32(PIC32MX250F128B)
#elif defined( __PIC32MX__ )
  #pragma config UPLLEN   = ON            // USB PLL Enabled
  #pragma config FPLLMUL  = MUL_15        // PLL Multiplier
  #pragma config UPLLIDIV = DIV_2         // USB PLL Input Divider
  #pragma config FPLLIDIV = DIV_2         // PLL Input Divider
  #pragma config FPLLODIV = DIV_1         // PLL Output Divider
  #pragma config FPBDIV   = DIV_1         // Peripheral Clock divisor
  #pragma config FWDTEN   = OFF           // Watchdog Timer
  #pragma config WDTPS    = PS1           // Watchdog Timer Postscale
    //#pragma config FCKSM    = CSDCMD        // Clock Switching & Fail Safe Clock Monitor
  #pragma config OSCIOFNC = OFF           // CLKO Enable
  #pragma config POSCMOD  = HS            // Primary Oscillator
  #pragma config IESO     = OFF           // Internal/External Switch-over
  #pragma config FSOSCEN  = OFF           // Secondary Oscillator Enable (KLO was off)
  #pragma config FNOSC    = PRIPLL        // Oscillator Selection
  #pragma config CP       = OFF           // Code Protect
  #pragma config BWP      = OFF           // Boot Flash Write Protect
  #pragma config PWP      = OFF           // Program Flash Write Protect
  #pragma config ICESEL   = ICS_PGx1      // ICE/ICD Comm Channel Select
#else
    #error Cannot define configuration bits.
#endif

FSFILE * myFile;
BYTE myData[512];
size_t numBytes;
volatile BOOL deviceAttached;

int main(void)
{
  unsigned long blink_counter;

  //ET-BASE PIC32(PIC32MX250F128B)
  #if defined(__PIC32MX__)
  {
    int  value;
    
    value = SYSTEMConfigWaitStatesAndPB( GetSystemClock() );
    
    // Enable the cache for the best performance
    CheKseg0CacheOn();
    
    INTEnableSystemMultiVectoredInt();
    
    value = OSCCON;
    while (!(value & 0x00000020))
    {
      value = OSCCON;    // Wait for PLL lock to stabilize
    }
  }
  #endif

  deviceAttached = FALSE;

  //Initialize the stack
  USBInitialize(0);

  #if defined(DEBUG_MODE)
  
    //ET-BASE PIC32(PIC32MX250F128B)Mapping RB8=RX2,RB9=TX2
    PPSInput(2,U2RX,RPB8);                                                        // Assign RPB8 as input pin for U2RX
    PPSOutput(4,RPB9,U2TX);                                                       // Set RPB9 pin as output for U2TX
    UART2Init();

  #endif

  //Initialize all of the LED pins
  mInitAllLEDs();

  //Start Operate Status
  mLED_1_On(); mLED_2_On();
  blink_counter = 0;
  
  while(1)
  {
    //USB stack process function
    USBTasks();

    blink_counter++; 
    if(blink_counter>(SYS_FREQ/250))
    { 
      blink_counter = 0;
      mLED_1_Toggle();
      mLED_2_Toggle();
    }

    //if thumbdrive is plugged in
    if(USBHostMSDSCSIMediaDetect())
    {
      deviceAttached = TRUE;

      //Connect USB MSD Device Status
      mLED_1_On(); mLED_2_On();

      //now a device is attached
      //See if the device is attached and in the right format
      if(FSInit())
      {
        // Start of Initial USB MSD Device Status
        mLED_1_On(); mLED_2_Off();

        // Opening a file in mode "w" will create the file if it doesn't
        // exist.  If the file does exist it will delete the old file
        // and create a new one that is blank.
        myFile = FSfopen("test.txt","w");

        // Write some data to the new file.
        FSfwrite("This is a test.",1,15,myFile);
                
        // Always make sure to close the file so that the data gets
        // written to the drive.
        FSfclose(myFile);

        // End of Write Text File to USB MSD Device Status
        mLED_1_Off(); mLED_2_On();

        // Just sit here until the device is removed.
        blink_counter = 0;
        while(deviceAttached == TRUE)
        {
          USBTasks();

          blink_counter++;
          if(blink_counter>(SYS_FREQ/250))
          {
            blink_counter = 0;
            mLED_1_Toggle();
            mLED_2_Toggle();
          }
        }

        // Disconnect USB MSD Device Status
        mLED_1_Off(); mLED_2_Off();
      }
    }
  }
  return 0;
}

/****************************************************************************
  Function:
    BOOL USB_ApplicationEventHandler( BYTE address, USB_EVENT event,
                void *data, DWORD size )

  Summary:
    This is the application event handler.  It is called when the stack has
    an event that needs to be handled by the application layer rather than
    by the client driver.

  Description:
    This is the application event handler.  It is called when the stack has
    an event that needs to be handled by the application layer rather than
    by the client driver.  If the application is able to handle the event, it
    returns TRUE.  Otherwise, it returns FALSE.

  Precondition:
    None

  Parameters:
    BYTE address    - Address of device where event occurred
    USB_EVENT event - Identifies the event that occured
    void *data      - Pointer to event-specific data
    DWORD size      - Size of the event-specific data

  Return Values:
    TRUE    - The event was handled
    FALSE   - The event was not handled

  Remarks:
    The application may also implement an event handling routine if it
    requires knowledge of events.  To do so, it must implement a routine that
    matches this function signature and define the USB_HOST_APP_EVENT_HANDLER
    macro as the name of that function.
  ***************************************************************************/

BOOL USB_ApplicationEventHandler( BYTE address, USB_EVENT event, void *data, DWORD size )
{
  switch( event )
  {
    case EVENT_VBUS_REQUEST_POWER:
      // The data pointer points to a byte that represents the amount of power
      // requested in mA, divided by two.  If the device wants too much power,
      // we reject it.
    return TRUE;

    case EVENT_VBUS_RELEASE_POWER:
      // Turn off Vbus power.
      // The PIC24F with the Explorer 16 cannot turn off Vbus through software.

      //This means that the device was removed
      deviceAttached = FALSE;
      return TRUE;
    break;

    case EVENT_HUB_ATTACH:
      return TRUE;
    break;

    case EVENT_UNSUPPORTED_DEVICE:
      return TRUE;
    break;

    case EVENT_CANNOT_ENUMERATE:
      //UART2PrintString( "\r\n***** USB Error - cannot enumerate device *****\r\n" );
      return TRUE;
    break;

    case EVENT_CLIENT_INIT_ERROR:
      //UART2PrintString( "\r\n***** USB Error - client driver initialization error *****\r\n" );
      return TRUE;
    break;

    case EVENT_OUT_OF_MEMORY:
      //UART2PrintString( "\r\n***** USB Error - out of heap memory *****\r\n" );
      return TRUE;
    break;

    case EVENT_UNSPECIFIED_ERROR:   // This should never be generated.
      //UART2PrintString( "\r\n***** USB Error - unspecified *****\r\n" );
      return TRUE;
    break;

    default:
    break;
  }
  return FALSE;
}
